<?php
/**
 * Plugin Name: Dynamic Blog Post
 * Description: Displays latest blog post on left and other posts on right with scrollable design. Shortcode: [breaking_news]. Also ensures featured image shows on single post pages.
 * Version: 1.2
 * Author: Luvcite
 */

if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

// Enqueue styles
function breaking_news_enqueue_assets() {
    wp_enqueue_style( 'fontawesome', 'https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.0/css/all.min.css' );
    wp_add_inline_style( 'fontawesome', breaking_news_custom_css() );
}
add_action( 'wp_enqueue_scripts', 'breaking_news_enqueue_assets' );

// Shortcode output
function breaking_news_shortcode() {
    ob_start();

    // Fetch latest post
    $main_post = new WP_Query([
        'posts_per_page' => 1,
        'post_status' => 'publish'
    ]);

    // Fetch side posts
    $side_posts = new WP_Query([
        'posts_per_page' => 3,
        'offset' => 1,
        'post_status' => 'publish'
    ]);
    ?>
    <div class="news-container">

        <!-- Main News -->
        <?php if ( $main_post->have_posts() ) : while ( $main_post->have_posts() ) : $main_post->the_post(); ?>
            <div class="main-news">
                <div class="main-news_fig">
                    <?php if ( has_post_thumbnail() ) : ?>
                        <?php the_post_thumbnail( 'large', ['alt' => get_the_title()] ); ?>
                    <?php else : ?>
                        <img src="https://via.placeholder.com/800x500?text=No+Image" alt="No Image">
                    <?php endif; ?>
                </div>
                <div class="main-news-content">
                    <div class="main-news-date">
                        <i class="fa-regular fa-calendar"></i> <?php echo get_the_date(); ?> 
                        | <span class="blog-name"><?php bloginfo('name'); ?></span>
                    </div>
                    <h2>
                        <a href="<?php the_permalink(); ?>" target="_blank"><?php the_title(); ?></a>
                    </h2>
                    <p><?php echo wp_trim_words( get_the_excerpt(), 30, '...' ); ?></p>
                    <a href="<?php the_permalink(); ?>" target="_blank" class="read-more">Read More</a>
                </div>
            </div>
        <?php endwhile; wp_reset_postdata(); endif; ?>

        <!-- Side News -->
        <div class="side-news">
            <?php if ( $side_posts->have_posts() ) : while ( $side_posts->have_posts() ) : $side_posts->the_post(); ?>
                <div class="news-item">
                    <a href="<?php the_permalink(); ?>" target="_blank">
                        <?php if ( has_post_thumbnail() ) : ?>
                            <?php the_post_thumbnail( 'thumbnail', ['alt' => get_the_title()] ); ?>
                        <?php else : ?>
                            <img src="https://via.placeholder.com/90x70?text=No+Image" alt="No Image">
                        <?php endif; ?>
                    </a>
                    <div class="news-info">
                        <div class="news-date">
                            <i class="fa-regular fa-calendar"></i> <?php echo get_the_date(); ?> 
                            | <span class="blog-name"><?php bloginfo('name'); ?></span>
                        </div>
                        <a href="<?php the_permalink(); ?>" target="_blank" class="news-title"><?php the_title(); ?></a>
                    </div>
                </div>
            <?php endwhile; wp_reset_postdata(); endif; ?>
        </div>

    </div>
    <?php
    return ob_get_clean();
}
add_shortcode( 'breaking_news', 'breaking_news_shortcode' );

// Force show featured image on single post pages
function breaking_news_add_featured_image_single( $content ) {
    if ( is_single() && in_the_loop() && is_main_query() ) {
        if ( has_post_thumbnail() ) {
            $featured = '<div class="single-featured-image">'
                        . get_the_post_thumbnail( get_the_ID(), 'large', ['class' => 'img-fluid', 'alt' => get_the_title()] )
                        . '</div>';
            $content = $featured . $content;
        }
    }
    return $content;
}
add_filter( 'the_content', 'breaking_news_add_featured_image_single' );

// CSS
function breaking_news_custom_css() {
return <<<CSS
body {
    font-family: "Open Sans", Sans-serif;
    margin: 0;
    padding: 0;
    background: #fff;
    color: #333;
}
.news-container {
    display: grid;
    grid-template-columns: 2fr 1fr;
    gap: 30px;
}
/* Main News */
.main-news {
    border-radius: 0px;
    overflow: hidden;
    box-shadow: 0 2px 6px rgba(0, 0, 0, 0.1);
    background: #fff;
    display: grid;
    grid-template-columns: 1fr 1fr;
}
.main-news_fig img {
    width: 100%;
    height: 100%;
    object-fit: cover;
    transition: transform 0.4s ease, box-shadow 0.3s ease;
}
.main-news_fig img:hover {
    transform: scale(1.05);
    box-shadow: 0 6px 14px rgba(0,0,0,0.2);
}
.main-news-content {
    padding: 20px;
}
.main-news-date {
    font-size: 14px;
    color: #666;
    margin-bottom: 8px;
}
.main-news-date i {
    margin-right: 6px;
    color: #00a6d6;
}
.blog-name {
    font-weight: bold;
    color: #007fa6;
}
.main-news h2 {
    margin: 0 0 15px;
    font-size: 24px;
    font-weight: bold;
    color: #000;
}
.main-news h2 a {
    text-decoration: none;
    color: inherit;
}
.main-news h2 a:hover {
    color: #00a6d6;
}
.main-news p {
    font-size: 15px;
    line-height: 1.6;
    color: #444;
}
.read-more {
    display: inline-block;
    margin-top: 15px;
    padding: 10px 20px;
    background: #00a6d6;
    color: #fff;
    border-radius: 20px;
    text-decoration: none;
    font-size: 14px;
    transition: background 0.3s ease;
}
.read-more:hover {
    background: #007fa6;
}
/* Side News */
.side-news {
    display: flex;
    flex-direction: column;
    gap: 20px;
//     max-height: 400px;
    max-height: auto;
    overflow-y: auto;
    padding-right: 5px;
}
.news-item {
    display: flex;
    gap: 12px;
    align-items: flex-start;
}
.news-item img {
    width: 125px;
    height: 70px;
    object-fit: cover;
    border-radius: 15px !important;
    flex-shrink: 0;
    transition: transform 0.4s ease, box-shadow 0.3s ease;
}
.news-item img:hover {
    transform: scale(1.05);
    box-shadow: 0 6px 14px rgba(0,0,0,0.2);
}
.news-info {
    flex: 1;
}
.news-date {
    font-size: 13px;
    color: #666;
    margin-bottom: 6px;
}
.news-date i {
    margin-right: 6px;
    color: #00a6d6;
}
.news-title {
    font-size: 22px;
    font-weight: bold;
    line-height: 1.3;
    color: #000;
    text-decoration: none;
}
.news-title:hover {
    color: #00a6d6;
}
/* Scrollbar */
.side-news::-webkit-scrollbar {
    width: 6px;
}
.side-news::-webkit-scrollbar-thumb {
    background: #ccc;
    border-radius: 10px;
}
.side-news::-webkit-scrollbar-thumb:hover {
    background: #00a6d6;
}
/* Featured Image on Single Post */
.single-featured-image {
    margin-bottom: 20px;
}
.single-featured-image img {
    width: 100%;
    height: auto;
    border-radius: 6px;
    display: block;
}
    .ast-narrow-container .site-content > .ast-container {
        max-width: 60%;
    }
.main-news_fig {
    overflow: hidden;
}



/* Responsive */
@media (max-width: 900px) {
    .news-container {
        grid-template-columns: 1fr;
    }
    .main-news {
        grid-template-columns: 1fr;
    }
    .main-news_fig img {
        height: 250px;
    }
    .side-news {
        max-height: unset;
        overflow-y: unset;
    }
}
@media (max-width: 600px) {
    .main-news-content { padding: 15px; }
    .main-news h2 { font-size: 20px; }
    .main-news p { font-size: 13px; }
    .news-item { flex-direction: column; }
    .news-item img { width: 100%; height: 180px; }
}
CSS;
}
