<?php
/*
Plugin Name: Custom Data Table
Description: A plugin to manage and display a scrollable responsive data table with admin panel control.
Version: 1.6
Author: You
*/

if ( ! defined( 'ABSPATH' ) ) exit;

// Enqueue Bootstrap + DataTables + custom CSS
function cdt_enqueue_scripts() {
    // Bootstrap
    wp_enqueue_style('bootstrap-css', 'https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css');

    // DataTables
    wp_enqueue_style('datatable-css', 'https://cdn.datatables.net/1.13.6/css/jquery.dataTables.min.css');
    wp_enqueue_script('jquery');
    wp_enqueue_script('datatable-js', 'https://cdn.datatables.net/1.13.6/js/jquery.dataTables.min.js', array('jquery'), null, true);

    // Initialize DataTable (no responsive collapse)
    wp_add_inline_script('datatable-js', '
    jQuery(document).ready(function($){
        $("#cdt-table").DataTable({
            responsive: false,
            scrollX: true,
            scrollCollapse: true,
            paging: true
        });
    });
    ');

    // Custom styling
    wp_add_inline_style('datatable-css', "
        #cdt-table tbody tr:hover { 
            background-color: #f1f9ff; 
            transition: background-color 0.3s ease; 
        }
        .cdt-actions a {
            margin: 0 4px;
            text-decoration: none;
        }
        .cdt-actions .dashicons {
            font-size: 18px;
            vertical-align: middle;
        }
        .cdt-actions .edit { color:#2271b1; }
        .cdt-actions .delete { color:#d63638; }

        /* Keep table grid on small screens */
        #cdt-table {
            width: 100% !important;
            white-space: nowrap;
        }
        .dataTables_wrapper {
            overflow-x: auto;
        }
    ");
}
add_action('wp_enqueue_scripts', 'cdt_enqueue_scripts');
add_action('admin_enqueue_scripts', 'cdt_enqueue_scripts');

// Create Admin Menu
function cdt_create_menu() {
    add_menu_page('Custom Data Table', 'Data Table', 'manage_options', 'cdt_admin', 'cdt_admin_page', 'dashicons-table-col-after', 20);
}
add_action('admin_menu', 'cdt_create_menu');

// Register custom table
function cdt_install() {
    global $wpdb;
    $table_name = $wpdb->prefix . 'cdt_entries';
    $charset_collate = $wpdb->get_charset_collate();

    $sql = "CREATE TABLE IF NOT EXISTS $table_name (
        id mediumint(9) NOT NULL AUTO_INCREMENT,
        first_name varchar(100) DEFAULT '',
        surname varchar(100) DEFAULT '',
        city varchar(100) DEFAULT '',
        postcode varchar(50) DEFAULT '',
        country varchar(100) DEFAULT '',
        telephone varchar(50) DEFAULT '',
        ap tinyint(1) DEFAULT 0,
        cap tinyint(1) DEFAULT 0,
        PRIMARY KEY (id)
    ) $charset_collate;";

    require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
    dbDelta($sql);
}
register_activation_hook(__FILE__, 'cdt_install');

// Admin Page Content
function cdt_admin_page() {
    global $wpdb;
    $table_name = $wpdb->prefix . 'cdt_entries';

    // Delete entry
    if (isset($_GET['delete'])) {
        $wpdb->delete($table_name, ['id' => intval($_GET['delete'])]);
        echo "<div class='updated'><p>Entry deleted!</p></div>";
    }

    // Add entry
    if (isset($_POST['cdt_add_entry'])) {
        $wpdb->insert($table_name, [
            'first_name' => sanitize_text_field($_POST['first_name']),
            'surname' => sanitize_text_field($_POST['surname']),
            'city' => sanitize_text_field($_POST['city']),
            'postcode' => sanitize_text_field($_POST['postcode']),
            'country' => sanitize_text_field($_POST['country']),
            'telephone' => sanitize_text_field($_POST['telephone']),
            'ap' => isset($_POST['ap']) ? 1 : 0,
            'cap' => isset($_POST['cap']) ? 1 : 0,
        ]);
        echo "<div class='updated'><p>Entry added!</p></div>";
    }

    echo '<div class="wrap"><h1>Manage Data Table</h1>';
    echo '<form method="post">
        <input type="text" name="first_name" placeholder="First Name">
        <input type="text" name="surname" placeholder="Surname">
        <input type="text" name="city" placeholder="City">
        <input type="text" name="postcode" placeholder="Postcode">
        <input type="text" name="country" placeholder="Country">
        <input type="text" name="telephone" placeholder="Telephone">
        <label><input type="checkbox" name="ap"> AP</label>
        <label><input type="checkbox" name="cap"> CAP</label>
        <input type="submit" name="cdt_add_entry" value="Add Entry" class="button button-primary">
    </form><br>';

    // Fetch entries
    $results = $wpdb->get_results("SELECT * FROM $table_name");
    if ($results) {
        echo '<div class="table-responsive">';
        echo '<table class="table table-bordered table-striped"><thead><tr>
            <th>First Name</th><th>Surname</th><th>City</th><th>Postcode</th>
            <th>Country</th><th>Telephone</th><th>AP</th><th>CAP</th><th>Actions</th></tr></thead><tbody>';
        foreach ($results as $row) {
            echo "<tr>
                <td>{$row->first_name}</td>
                <td>{$row->surname}</td>
                <td>{$row->city}</td>
                <td>{$row->postcode}</td>
                <td>{$row->country}</td>
                <td>{$row->telephone}</td>
                <td>" . ($row->ap ? '⭐' : '☆') . "</td>
                <td>" . ($row->cap ? '⭐' : '☆') . "</td>
                <td class='cdt-actions'>
                    <a href='?page=cdt_admin&delete={$row->id}' title='Delete' onclick='return confirm(\"Delete this entry?\");'><span class='dashicons dashicons-trash delete'></span></a>
                </td>
            </tr>";
        }
        echo '</tbody></table>';
        echo '</div>';
    }
    echo '</div>';
}

// Shortcode for frontend display
function cdt_shortcode() {
    global $wpdb;
    $table_name = $wpdb->prefix . 'cdt_entries';
    $results = $wpdb->get_results("SELECT * FROM $table_name");

    ob_start();
    echo '<div class="table-responsive">';
    echo '<table id="cdt-table" class="table table-bordered table-striped display nowrap" style="width:100%"><thead><tr>
        <th>First Name</th><th>Surname</th><th>City</th><th>Postcode</th>
        <th>Country</th><th>Telephone</th><th>AP</th><th>CAP</th></tr></thead><tbody>';
    foreach ($results as $row) {
        echo "<tr><td>{$row->first_name}</td><td>{$row->surname}</td><td>{$row->city}</td>
        <td>{$row->postcode}</td><td>{$row->country}</td><td>{$row->telephone}</td>
        <td>" . ($row->ap ? '⭐' : '☆') . "</td><td>" . ($row->cap ? '⭐' : '☆') . "</td></tr>";
    }
    echo '</tbody></table>';
    echo '</div>';
    return ob_get_clean();
}
add_shortcode('custom_data_table', 'cdt_shortcode');
