/*
* v2.0
* Unlimited Video On Hover script made by Adarsh Pawar for Unlimited Elements for Elementor.
* © Unlimited Elements for Elementor, Adarsh Pawar.
* ------------------------------------------------------------------------------------------
* v2.0 (Improvement: Replaced touchstart with pointerup to allow scrolling on touch devices while still supporting tap-to-toggle video)
* v1.9 (Improved UX: For smoother experience, delayed video reset after hover out to prevent sudden jump to first frame)
* v1.8 (Feature: Added data-debug="true" to show content as if hovered, useful for designing)
* v1.7 (Feature: Added preload optimization using IntersectionObserver for HTML5 video to load metadata offscreen and buffer when in view)
* v1.6 (Fix: HTML5 video requiring two hovers if mouse was already over the widget)
* v1.5 (Feature: Added option to display overlay only on touch devices or non-touch devices)
* v1.4 (Improvement: Autoplay handling for touch vs. non-touch devices)
*/

(function (jQuery) {
    function isTouchDevice() {
        return 'ontouchstart' in window || navigator.maxTouchPoints > 0;
    }

    window.UEVideoHoverWidget = function (objWidget) {
        var image = objWidget.find('.ue-image')[0];
        var video = objWidget.find('.ue-video')[0];

        if (!image || !video || video.dataset.initialized === 'true') return;

        var isVimeo = video.tagName === 'IFRAME' && video.src.includes('vimeo.com');
        var vimeoPlayer = null;
        var vimeoReady = false;
        var isHovered = false;
        var isDebug = objWidget.attr('data-debug') === 'true';

        var isMouseAlreadyOver = !isTouchDevice() && (objWidget.is(':hover') || isDebug);

        let resetTimeout;

        function showVideo() {
            if (resetTimeout) {
                clearTimeout(resetTimeout);
                resetTimeout = null;
            }

            if (isVimeo && !vimeoReady) return;
            if (!isVimeo && video.readyState < 3) return;

            objWidget.addClass('ue-video-playing');
            image.style.opacity = 0;

            if (isVimeo) {
                vimeoPlayer.play();
            } else {
                video.play();
            }
        }

        function hideVideo() {
            objWidget.removeClass('ue-video-playing');
            image.style.opacity = 1;

            if (isVimeo) {
                vimeoPlayer.pause().then(function () {
                    resetTimeout = setTimeout(() => {
                        vimeoPlayer.setCurrentTime(0.01);
                    }, 300);
                });
            } else {
                video.pause();
                resetTimeout = setTimeout(() => {
                    video.currentTime = 0;
                }, 300);
            }
        }

        if (!isVimeo) {
            video.muted = true;
            video.preload = 'metadata';

            if ('IntersectionObserver' in window) {
                var observer = new IntersectionObserver(function (entries, obs) {
                    entries.forEach(function (entry) {
                        if (entry.isIntersecting) {
                            video.preload = 'auto';
                            video.load();
                            obs.unobserve(entry.target);
                        }
                    });
                }, {
                    rootMargin: '0px',
                    threshold: 0.25
                });

                observer.observe(video);
            } else {
                video.preload = 'auto';
            }

            video.addEventListener('canplay', function () {
                video.dataset.ready = 'true';
                if (isHovered || isMouseAlreadyOver) {
                    isHovered = true;
                    showVideo();
                }
            });
        } else {
            vimeoPlayer = new Vimeo.Player(video);

            vimeoPlayer.setCurrentTime(0.01).then(function () {
                return vimeoPlayer.pause();
            }).then(function () {
                vimeoReady = true;
                if (isHovered || isMouseAlreadyOver) {
                    isHovered = true;
                    showVideo();
                }
            });
        }

        if (!isTouchDevice() && !isDebug) {
            objWidget.on('mouseenter', function () {
                isHovered = true;
                showVideo();
            });

            objWidget.on('mouseleave', function () {
                isHovered = false;
                hideVideo();
            });
        } else if (!isDebug) {
            objWidget.addClass('touch-device');

            objWidget.on('pointerup', function (e) {
                if (e.pointerType !== 'touch' && e.pointerType !== 'pen') return;

                if (isVimeo) {
                    vimeoPlayer.getPaused().then(function (paused) {
                        if (paused) {
                            showVideo();
                        } else {
                            hideVideo();
                        }
                    });
                } else {
                    if (video.paused && video.dataset.ready === 'true') {
                        showVideo();
                    } else {
                        hideVideo();
                    }
                }

                e.stopPropagation();
            });

            jQuery(document).on('pointerup', function (e) {
                if (!jQuery(e.target).closest(objWidget).length) {
                    hideVideo();
                }
            });
        }

        video.dataset.initialized = 'true';

        var resizeTimeout;
        jQuery(window).on('resize', function () {
            clearTimeout(resizeTimeout);
            resizeTimeout = setTimeout(function () {
                UEVideoHoverWidget(objWidget);
            }, 300);
        });
    };

    if (window.location.href.includes('&preview=true')) {
        console.log(
            `%c✨ Video on Hover v2.0 ready!`,
            'color: #00c896; font-weight: bold; font-size: 12px;'
        );
    }
})(jQuery);
